#' Integral Condition Index Weighting
#'
#' Consensuates and scales the weights of some variables on a complex construct.
#'
#' @param data Excel book with its extension.
#' @param median Calculate median weights (def. median = T) or mean weights (median = F).
#' @param ... Additional argument when data is read from Google Sheets.
#'    ID: Id of the Google Sheets. cons: Create one sheet with the consensus weights (def cons = T).
#' @return Consensus median (or mean) weights.
#' @author Guillermo Correa (gcorrea@unal.edu.co)
#' @export
#' @references Correa-Londoño, G. (2023). Índice de Condición Integral. https://bookdown.org/gcorrea/ICI/.
#' @seealso [ici] for calculating the ICI, and [ici_escal] for consensuating and
#'   scaling the category scores of a variable.
#' @examples
#' # NOTICE: Given that Excel files are not R objects, but external files, they cannot be
#' # loaded into memory when ici package is attached. So, in order to run examples
#' # you must make Excel files available by locating them on your working directory path.
#' # You can find Excel files in the 'extdata/' directory within the installed package
#' # on your local drive. You can also download them through the book:
#' # https://bookdown.org/gcorrea/ICI/ICIresources.rar.
#' #
#' # Example 1: Weights based on median (default)
#' ici_pond("ici_pond.xlsx")
#'
#' # Example 2: Weights based on mean (check book for details)
#' ici_pond("ici_pond.xlsx", median = F)
#'
#' # Example 3: Equiponderated variables
#' ici_pond("ici_pond.xlsx", sheet = 3)
#'
#' # Example 4: Equiponderated variables with some zero-weight variables
#' ici_pond("ici_pond.xlsx", sheet = 4)
#'
#' # Example 5: Some zero-weight variables
#' ici_pond("ici_pond.xlsx", sheet = 5)
ici_pond <- function (data = NULL, median = T, ...)
{
  # Lectura desde una hoja de cálculo de Google --------------------------------
  if (is.null(data)) {
    args   <- list(...)
    ID     <- args$ID
    if(length(args) > 1)
      cons <- args$cons
    else
      cons <- T
    exp    <- length(googlesheets4::gs4_get(ID)[[6]][[1]])
    if((googlesheets4::gs4_get(ID)[[6]][[1]][[exp]]) == "consensus weight")
      exp  <- exp - 1
    for (i in 1:exp) {
      if (i == 1) {
        data <- googlesheets4::read_sheet(ID, 1, range = "A:B")
      }
      else
        data[, i + 1] <- googlesheets4::read_sheet(ID, i, range = "A:B")[[2]]
    }
  }

  # Lectura desde un libro de Excel --------------------------------------------
  else {
    args   <- list(...)
    sheet  <- args$sheet
    if(length(args) > 1)
      cons <- args$cons
    else
      cons <- T

    if(is.null(sheet))
      sheet <- 1
    data <- readxl::read_excel(data, sheet)
  }

  # Estandarización y cálculo de ponderaciones consenso ------------------------
  var.name <- names(data)[[1]]
  data.calc <- data[, -1]
  medias <- sapply(data.calc, mean, na.rm = T)
  data.calc <- data.calc/rep(medias, each = nrow(data))
  var.name <- names(data)[[1]]
  if (median)
    data <- data.frame(data[[1]], (apply(data.calc, 1, stats::median, na.rm = T)))
  else
    data <- data.frame(data[[1]], (apply(data.calc, 1, mean, na.rm = T)))

  # Escalamiento de las ponderaciones ------------------------------------------
  # Extracción de ponderaciones cero
  ceros <- NULL
  if (min(data[[2]]) == 0) {
    ceros <- (data[[2]] == 0)
    data[ceros, 2] <- NA
  }

  # Escalamiento con ponderación mínima 1
  p.min <- min(data[[2]], na.rm = T)
  data[[3]] <- data[[2]]/p.min

  # Escalamiento con ponderación máxima 5
  p.max <- max(data[[2]], na.rm = T)
  data[[4]] <- 5 * data[[2]]/p.max

  # Escalamiento ponderaciones porcentuales
  p.sum <- sum(data[[2]], na.rm = T)
  data[[5]] <- 100 * data[[2]]/p.sum

  # Reincorporación de ponderaciones cero y retiro de ponderaciones sin escalar
  if(!is.null(ceros))
    data[ceros, c(3, 4, 5)] <- 0
  data <- data[, -2]

  # Salidas --------------------------------------------------------------------
  names(data) <- c(var.name, "min1 wg", "max5 wg", "percent wg")
  print(data, row.names = F, print.gap = 3, digits = 3)
  if(exists("ID") & cons) {
    googlesheets4::sheet_write(data, ID, sheet = "consensus weight")
  }
  output <- data
}
######################        Fin de la Función       #########################|
